# Agent Based Model - Z/Yen - ver 3.0

# In this model there is a cryptocurrency (CC) and an established fiat currency (FC). We are focusing on the
# first year(s) of the cryptocurrency and imagine that the fiat currency is stable, and with no changes in
# its prices, interest rates, etc.
#
# You can run an individual trial of multiple days. (Set the parameter below.)
# The days list then holds the result of all the days.
#
# You can run multiple trials, each of multiple days. (Again, set the parameters below.)
# The trials list then holds the result of all the trials. At present, only the final FC/CC rate.


# PACKAGES TO LOAD

library(ineq)

# PARAMETERS

# initial set up
r_seed <- -1			# random seed; -1 means use the default
m_max <- 10  			# number of merchants
m_gnave <- 5			# average number of goods offered per merchant
m_gnineq <- 1			# inequality factor; the s in 1/k^s. 0.25 is almost flat, 4 is almost all goods for first supplier
m_pave <- 3				# average price of first good for a merchant
m_pineq <- 1			# inequality factor for price of first good for each merchant
c_max <- 40				# number of customers
c_pave <- 5				# average number of purchases per day for each customer
c_pineq <- 1			# inequality factor for number of purchases per day per customer; 0.25 is nearly flat; 4 is very, very skewed. The s in 1/k^s.
ccr_start <- 1			# initial FC/CC rate i.e. FC per CC, the price of one unit of CC in NCs
cc_fxl <- 0.03			# fraction of currency lost in a round trip (FC to CC to FC) due to bid-ask spread
ccm_start <- 10			# initial minting cost of a unit of CC, expressed in FC
pool_start <- 10000 		# initial CC pool held by the currency exchange
buzz_start <- 10			# initial buzz level
t_days <- 365			# number of days to simulate in one trial
s_trials <- 100			# number of trials to use in MC simulation
fmask <- c(1,1,1,1,1,1)		# mask for factors; 0 will take the factor out of consideration
rcncy <- c(0,0.1)			# range for recency factors

# exchange parameters
exc_type <- 9			# exchange mechanism choice
exc_width <- 12000		# width parameter over which multiplier varies - intended range of pool
exc_greatest <- 5			# controls max size of multiplier (at zero)
exc_peak <- 200			# also controls max size of multiplier (at zero)
exc_damp <- 2000			# overall strength of multiplier over whole range

# merchant decisions
#m_in_prob <- 0.1			# replaced/ probability of going from non participation to participation on a day
#m_out_prob <- 0.1		# replaced/ probability of goind from participation back to non-participation on a day
m_now_prob <- 0.1			# probability of choosing to set 'now' prices, given opting in
m_wkly_prob <- 0.2		# probability of choosing weekly prices, given not setting 'now' prices (alternative is monthly prices)
m_lambda <- 0.3			# lambda for EWMA of sentiment
m_unitsales <- 10			# daily sales in CC prices that yields sentiment of 1 for the day
m_unitbuzz <- 10			# level of buzz that yields sentiment of 1 for the day
m_sbfrac <- 0.5			# proportion of sentiment contributed by sales each day
m_pot_mult <- 10			# multiplier used to set merchants' cashpots

# customer decisions
#c_in_prob <- 0.1			# replaced/ probability of going from non participation to participation on a day
#c_out_prob <- 0.1		# replaced/ probability of goind from participation back to non-participation on a day
c_lambda <- 0.1			# lambda for EWMA used for sentiment
c_unitsav <- 10			# daily savings through choosing CC prices that yields sentiment of 1 for the day
c_unitbuzz <- 10			# level of buzz that yields sentiment of 1 for the day
c_sbfrac <- 0.5			# proportion of sentiment contributed by savings each day
c_pot_mult <- 20			# multiplier used to set customers' cashpots


trials1 <- data.frame()
trials2 <- data.frame()


# Other global objects

params <- data.frame(
	varname = character(),	# variable name in the code
	desc = character(),	# its description
	val = character(),	# its value, converted to characters if necessary
	stringsAsFactors=FALSE
)

merchants <- data.frame(
	id = numeric(), 		# merchant id
	weights = list(),		# vector of weights for factors in evaluating the CC
	lambdas = list(),		# vector of lambdas used as recency factors in EWMAs
	ewmas = list(),		# vector of EWMA numbers for the factors used in evaluation
	cashpot = numeric(),	# amount of cash the be split between CC and FC
	ngo = numeric(),		# number of goods offered
	ng1 = numeric(), 		# number of first good
	ngn= numeric(),		# number of last good
	hip = numeric(), 		# highest price of a good
	prd = numeric(),		# price revision day, used for weekly/monthly repricing
	daysales = numeric(),	# total sales of the day in CC, but valued in FC
	ccsales = numeric(), 	# total sales of the day in CC, valued in CC
	prevccsales = numeric(),	# total sales of the previous day in CC, valued in CC
	sent = numeric(),		# sentiment towards CC
	incc = numeric(), 	# in or out of CC
	meth = character(), 	# method of price setting
	cc = numeric(),		# balance of CC at end of day
	stringsAsFactors=FALSE
)

goods <- data.frame(
	id=numeric(),		# good id
	mid=numeric(),		# merchant id
	ncp=numeric(),		# FC price
	ccp=numeric(),		# CC price, if there is one
	stringsAsFactors=FALSE
)
	
customers <- data.frame(
	id=numeric(),		# customer id
	weights = list(),		# vector of weights for factors in evaluating the CC
	lambdas = list(),		# vector of lambdas used as recency factors in EWMAs
	ewmas = list(),		# vector of EWMA numbers for the factors used in evaluation
	cashpot = numeric(),	# amount of cash the be split between CC and FC
	ppd=numeric(),		# average purchases per day
	sent=numeric(),		# sentiment towards CC, +ve or -ve
	incc=numeric(),		# in or out of CC 
	ccsavings = numeric(), 	# total savings of the day by buying in CC, valued in CC
	prevccsavings = numeric(),	# total savings of the previous day, valued in CC
	cc=numeric()		# CC balance
)

days <- data.frame(
	id=numeric(),		# day id
	ccr=numeric(),		# f4: FC/CC rate
	ccm=numeric(),		# minting cost of a unit of CC, expressed in FC
	merin=numeric(),		# number of merchants participating
	cusin=numeric(),		# number of customers participating
	fcpurch=numeric(),	# value of purchases made in FC, in NCs
	ccpurch=numeric(),	# value of purchases made in CC, in CC
	purchRE=numeric(),	# relative entropy of commerce activity i.e. purchases in CC
	purchbyRE=numeric(),	# f2: commerce activity multiplied by relative entropy
	purchGI=numeric(),	# 1- Gini index of commerce activity i.e. purchases in CC
	purchbyGI=numeric(),	# commerce activity multiplied by 1 - Gini index
	ccpurchatncp=numeric(),	# value of purchases made in CC, but using the FC prices also offered to give FC value
	pool=numeric(),		# CC pool quantity
	mercc=numeric(),		# total CC value held at the end of the day by merchants
	cuscc=numeric(),		# total CC value held at the end of the day by customers
	totalcc=numeric(),	# total CC everywhere at the end of the day
	heldRE=numeric(),		# relative entropy of CC holdings
	ccheldbyRE=numeric(),	# f3: holdings of CC multiplied by relative entropy
	heldGI=numeric(),		# 1 - Gini index of CC holdings
	ccheldbyGI=numeric(),	# holdings of CC multiplied by 1 - Gini index
	velcc=numeric(),		# velocity of CC each day calculated using CC only
	REvelcc=numeric(),	# velocity of CC calculated using Entropy distribution adjustment
	GIvelcc=numeric(),	# velocity of CC calculated using 1 - Gini index as the distribution adjustment
	fisherccr=numeric(),	# calculated exchange rate for CC using modified Fisher equation, if possible
	mermint=numeric(),	# total CC minted by merchants
	cusmint=numeric(),	# total CC minted by customers
	merccex=numeric(),	# CC exchanged by merchants +ve means bought, -ve means sold
	cusccex=numeric(),	# CC exchanged by customers +ve means bought, -ve means sold
	excRE=numeric(),		# relative entropy of exchange activity
	excbyRE=numeric(),	# f1: exchange activity multiplied by relative entropy
	excGI=numeric(),		# 1 - Gini index of exchange activity
	excbyGI=numeric(),	# exchange activity multiplied by 1 - Gini index
	buzz=numeric()		# buzz for the day
)

purchases <- list()		# list for lists of purchases of goods each day

exchanges <- list()		# list for lists of exchanges of goods each day

trials <- data.frame(
	id=numeric(),		# trial number
	fccr=numeric(),		# final FC/CC
	poolmax=numeric(),	# maximum value of pool
	poolmin=numeric(),	# minimum value of pool
	totalcc=numeric(),	# final total CC
	maxccr=numeric(),		# max ccr
	minccr=numeric(),		# min ccr
	aveccr=numeric(),		# mean ccr
	sd=numeric(),		# sample standard deviation of ccr
	fd=numeric(),		# fractal dimension of ccr
	runave=numeric(),		# mean value of absolute of runs
	diffave=numeric(), 	# average absolute daily difference
	poolout=numeric()		# 1 if a pool out occurred, 0 otherwise
)

expts <- data.frame(
	id=numeric(),		# experiment number
	mfccr=numeric()		# mean final FC/CC
)


ccr <- ccr_start		# FC/CC rate for the day
ccm <- ccm_start		# minting rate for the day
ccpool <- pool_start	# pool of CC held by fx market maker
buzz <- buzz_start	# buzz level at start
aveprice <- 0		# average price of goods on offer, in FC
seedstore <- c()		# vector to store the random seed if necessary
prevr_seed <- -1		# previous value of r_seed; -1 means use default


# FUNCTIONS

# functions for analysing time series

relative_entropy <- function(ar) {
	# returns the relative entropy of ar, a vector of non-negative numbers
	# entropy of even spread is log2(n)
	# entropy of vector is sum(-di/S*log2(di/S)) with p.log(p) = 0 for p = 0
	n <- length(ar)
	if (n == 0) {
		return(NaN)
	} else if (n == 1) {
		return(1)
	} else {
		n <- length(ar)
		S <- sum(ar)
		h <- sum(ifelse(ar == 0, 0, -ar*log2(ar/S)/S)/log2(n))
		return(h)
	}
}


higuchi=function(data,kmax)
{
  # Higuchi's algorithm; calculation heavy but very simple and extremely accurate
  # data should be a column vector of values
  # Hector's version
  
  N=NROW(data)
  Lmean=rep(0,kmax)
  for (k in c(1:kmax))
  {
    L=rep(0,k)
    for (m in c(1:k))
    {
      v=floor((N-m)/k)
      for (i in c(1:v))
      {
        L[m]=L[m]+abs(data[m+i*k]-data[m+(i-1)*k])
      }
      L[m]=L[m]*((N-1)/(v*k))*(1/k)
    }
    Lmean[k]=mean(L)
  }
   

  D=c(1:kmax)
  D=log(1/D[1:kmax])
  Llog=log(Lmean[1:kmax])
  mod=lm(Llog~D,weights=(1:kmax)^(-2.5))

  #uncomment for some plots
  #layout(matrix(c(1,1,1,1,2,3,4,5),2,4,byrow=FALSE))
  plot(Llog~D)
  abline(mod$coefficients)
  #plot(mod)
  return(mod$coefficients[2])
  
}


run_dist <- function(x) {
	# returns a vector that is the absolute value of movements up and down
	# Used to analyse the distribution of lengths of runs up and down.
	direction <- "none"
	moves <- c()
	mn <- 0
	diff <- 0
	start_val <- x[1]
	for (i in 2:length(x)) {
		diff <- x[i] - x[i-1]
		if (direction == "none") {
			if (diff == 0) {
				direction <- "flat"
			} else if (diff > 0) {
				direction <- "rising"
			} else {
				direction <- "falling"
			}
		} else if (direction == "flat") {
			if (diff != 0) {
				mn <- mn + 1
				moves[mn] <- 0
				start_val <- x[i-1]
				if (diff < 0) {
					direction <- "falling"
				} else {
					direction <- "rising"
				}
			}
		} else if (direction == "rising") {
			if (diff <= 0) {
				mn <- mn + 1
				moves[mn] <- x[i-1] - start_val
				start_val <- x[i-1]
				if (diff < 0) {
					direction <- "falling"
				} else {
					direction <- "flat"
				}
			}
		} else if (direction == "falling") {
			if (diff >= 0) {
				mn <- mn + 1
				moves[mn] <- start_val - x[i-1]
				start_val <- x[i-1]
				if (diff > 0) {
					direction <- "rising"
				} else {
					direction <- "flat"
				}
			}
		}
	}
	mn <- mn + 1
	if (direction == "flat") {
		moves[mn] <- 0
	} else if (direction == "rising") {
		moves[mn] <- x[length(x)] - start_val
	} else {
		moves[mn] <- start_val - x[length(x)]
	}
	return(moves)
}


# functions for initialisation

int_dist <- function(L, N, S) {    # returns an array L long, of integers adding up to (approximately) N, with Zeta distribution parameter S
	A <- 10 * N/L
	na <- 1:L
	na <- A/(na^S)
	na <- ifelse(na < 1, 1, ceiling(na))
	t <- sum(na)
	na <- na*N/t
	na <- ifelse(na < 1, 1, ceiling(na))
	return(na)
}

expt_reset <- function() {	# resets for new experiment
	expts <<- expts[0,]
}

sim_reset <- function() {  	# resets for a new MC simulation
	trials <<- trials[0,]
}

trial_reset <- function() {   # resets for a new trial i.e. doing the specified number of days once
	aveprice <<- 0		# will initially accumulate total price of goods listed, then divide through
	ga <- c()
	pa <- c()
	# set random seed if necessary
	if (r_seed == -1) {
		# will use default this time
		if (prevr_seed != -1) {
			# not default last trial so restore default stored
			.Random.seed <- seedstore
		}
	} else {
		# not using default seed, but instead will use seed provided in r_seed
		if (prevr_seed == -1) {
			# used default last time so store it for possible restoration later
			seedstore <- .Random.seed
		}
		set.seed(r_seed)
	}
	prevr_seed <- r_seed
	# set up merchants
	merchants <<- merchants[0,]
	g <- 0
	ga <- int_dist(m_max, m_gnave*m_max, m_gnineq)			# number of goods offered by each merchant
	pa <- int_dist(m_max, m_pave*m_max*100, m_pineq)/100		# price of first good offered by each merchant, to nearest penny
	for (n in 1:m_max) {
		merchants[n,1] <<- n						# using [n,1] format adds row safely to data.frame - a workaround
		merchants$weights[[n]] <<- runif(6)
		merchants$lambdas[[n]] <<- runif(6, min=rcncy[1], max=rcncy[2])
		merchants$ewmas[[n]] <<- c(0,0,0,0,0,0)
		merchants$ngo[n] <<- ga[n]
		merchants$ng1[n] <<- g + 1
		merchants$ngn[n] <<- g + ga[n]
		g <- g + ga[n]
		merchants$hip[n] <<- pa[n]
		merchants$cashpot[n] <<- merchants$ngo[n] * merchants$hip[n] * m_pot_mult
		merchants$prd[n] <<- 0
		merchants$meth[n] <<- "out"
		merchants$daysales[n] <<- 0
		merchants$ccsales[n] <<- 0
		merchants$prevccsales[n] <<- 0
		merchants$sent[n] <<- 0
		merchants$incc[n] <<- 0
		merchants$cc[n] <<- 0
	}
	# set up goods
	goods <<- goods[0,]
	gs <- 0
	gf <- 0
	for (m in 1:m_max) {
		gs <- merchants$ng1[m]
		gf <- merchants$ngn[m]
		for (g in gs:gf) {
			goods[g,1] <<- g
			goods$mid[g] <<- merchants$id[m]
			goods$ncp[g] <<- round(merchants$hip[m]/(g-gs+1), 2)
			goods$ccp[g] <<- 0
			aveprice <<- aveprice + goods$ncp[g]
		}
	}
	# calculate average price of goods (in FC) - needed for setting up customers
	aveprice <<- aveprice / merchants$ngn[m_max]
	# set up customers
	pa <- int_dist(c_max, c_pave*c_max*100, c_pineq)/100		# ave purchases per day to the cent
	customers <<- customers[0,]
	for (c in 1:c_max) {
		customers[c,1] <<- c
		customers$weights[[c]] <<- runif(6)
		customers$lambdas[[c]] <<- runif(6, min=rcncy[1], max=rcncy[2])
		customers$ewmas[[c]] <<- c(0,0,0,0,0,0)
		customers$ppd[c] <<- pa[c]
		customers$cashpot[c] <<- customers$ppd[c] * aveprice * c_pot_mult
		customers$sent[c] <<- 0
		customers$incc[c] <<- 0
		customers$ccsavings[c] <<- 0
		customers$prevccsavings[c] <<- 0
		customers$cc[c] <<- 0
	}
	# initialise days
	days <<- days[0,]
	# initialise purchases lists
	purchases <<- list()
	# initialise exchanges lists
	exchanges <<- list()
	# initialise other parameters
	ccr <<- ccr_start
	ccm <<- ccm_start
	ccpool <<- pool_start
	buzz <<- buzz_start
	# record parameter values
	params <<- params[0,]
	i <- 1
	params[i,1] <<- "r_seed"
	params[i,2] <<- "Random number seed - or -1 for default"
	params[i,3] <<- as.character(r_seed)
	i <- i + 1
	params[i,1] <<- "m_max"
	params[i,2] <<- "Number of merchants"
	params[i,3] <<- as.character(m_max)
	i <- i + 1
	params[i,1] <<- "m_gnave"
	params[i,2] <<- "Average number of goods offered by each merchant"
	params[i,3] <<- as.character(m_gnave)
	i <- i + 1
	params[i,1] <<- "m_gnineq"
	params[i,2] <<- "Inequality of number of goods offered by each merchant"
	params[i,3] <<- as.character(m_gnineq)
	i <- i + 1
	params[i,1] <<- "m_pave"
	params[i,2] <<- "Average price of first good for a merchant"
	params[i,3] <<- as.character(m_pave)
	i <- i + 1
	params[i,1] <<- "m_pineq"
	params[i,2] <<- "Inequality factor for price of first good for each merchant"
	params[i,3] <<- as.character(m_pineq)
	i <- i + 1
	params[i,1] <<- "c_max"
	params[i,2] <<- "Number of customers"
	params[i,3] <<- as.character(c_max)
	i <- i + 1
	params[i,1] <<- "c_pave"
	params[i,2] <<- "Mean purchases per day of each customer"
	params[i,3] <<- as.character(c_pave)
	i <- i + 1
	params[i,1] <<- "c_pineq"
	params[i,2] <<- "Inequality of number of purchases per day per customer"
	params[i,3] <<- as.character(c_pineq)
	i <- i + 1
	params[i,1] <<- "m_pot_mult"
	params[i,2] <<- "Multiplier used to set merchants' cashpots"
	params[i,3] <<- as.character(m_pot_mult)
	i <- i + 1
	params[i,1] <<- "c_pot_mult"
	params[i,2] <<- "Multiplier used to set customers' cashpots"
	params[i,3] <<- as.character(c_pot_mult)
	i <- i + 1
	params[i,1] <<- "ccr_start"
	params[i,2] <<- "Initial FC/CC exchange rate (mid point)"
	params[i,3] <<- as.character(ccr_start)
	i <- i + 1
	params[i,1] <<- "cc_fxl"
	params[i,2] <<- "Fraction of currency lost in a round trip"
	params[i,3] <<- as.character(cc_fxl)
	i <- i + 1
	params[i,1] <<- "ccm_start"
	params[i,2] <<- "Initial minting cost of a unit of CC, expressed in FC"
	params[i,3] <<- as.character(ccm_start)
	i <- i + 1
	params[i,1] <<- "pool_start"
	params[i,2] <<- "Initial CC pool held by the currency exchange"
	params[i,3] <<- as.character(pool_start)
	i <- i + 1
	params[i,1] <<- "t_days"
	params[i,2] <<- "Number of days to do in one trial"
	params[i,3] <<- as.character(t_days)
	i <- i + 1
	params[i,1] <<- "s_trials"
	params[i,2] <<- "Number of trials to do in one MC Simulation"
	params[i,3] <<- as.character(s_trials)
	i <- i + 1
	params[i,1] <<- "m_now_prob"
	params[i,2] <<- "Probability of setting 'now' prices, given opting in"
	params[i,3] <<- as.character(m_now_prob)
	i <- i + 1
	params[i,1] <<- "m_wkly_prob"
	params[i,2] <<- "Probability of setting weekly prices, given not setting 'now' prices"
	params[i,3] <<- as.character(m_wkly_prob)
	i <- i + 1
	params[i,1] <<- "fmask"
	params[i,2] <<- "Mask for including factors in the probability judgements"
	params[i,3] <<- paste("[", paste(fmask[1],fmask[2],fmask[3],fmask[4],fmask[5],fmask[6], sep=","),"]",sep="")
	i <- i + 1
	params[i,1] <<- "rcncy"
	params[i,2] <<- "Range for lambdas of merchants and customers"
	params[i,3] <<- paste("[", paste(rcncy[1],rcncy[2], sep=","),"]",sep="")
	i <- i + 1
	params[i,1] <<- "exc_type"
	params[i,2] <<- "Exchange rate revision method"
	params[i,3] <<- as.character(exc_type)
	i <- i + 1
	params[i,1] <<- "exc_width"
	params[i,2] <<- "Intended range of exchange's pool"
	params[i,3] <<- as.character(exc_width)
	i <- i + 1
	params[i,1] <<- "exc_greatest"
	params[i,2] <<- "Controls multiplier at pool=0"
	params[i,3] <<- as.character(exc_greatest)
	i <- i + 1
	params[i,1] <<- "exc_peak"
	params[i,2] <<- "Also controls multiplier at pool=0"
	params[i,3] <<- as.character(exc_peak)
	i <- i + 1
	params[i,1] <<- "exc_damp"
	params[i,2] <<- "Exchange rate revision speed attentuation"
	params[i,3] <<- as.character(exc_damp)
}

# functions to execute a day

ewma_delta <- function(x, lambda, type) {
# Return a vector that is the moving average of differences of x, exponentially weighted with lambda,
# either with arithmetic or percentage differences used.
	e <- x
	for (i in 1:length(x)) {
		if (i == 1) {
			e[i] <- 0
		} else if (i == 2) {
			e[i] <- x[2] - x[1]
		} else {
			e[i] <- lambda*(x[i]-x[i-1]) + (1 - lambda)*e[i-1]
		}
	}
	return(e)
}

purchase_count <- function(avep, maxp) { # a natural number between 0 and maxp inclusive, with an average of avep
	count <- 0
	for (i in 1: maxp) {
		if (runif(1) < avep/maxp) {
			count <- count + 1
		}
	}
	return(count)
}

delete_cc_prices <- function(mid) {	# delete all CC prices for merchant mid
	gs <- merchants$ng1[mid]
	gf <- merchants$ngn[mid]
	for (g in gs:gf) {
		goods$ccp[g] <<- 0
	}
}

set_cc_prices <- function(mid, x) {	# set all CC prices for merchant mid using exchange rate x; should be selling rate
	gs <- merchants$ng1[mid]
	gf <- merchants$ngn[mid]
	for (g in gs:gf) {
		goods$ccp[g] <<- max(0, goods$ncp[g]/x)
	}
}

update_ccr <- function(ccr, ex, pool, type) {
	# returns revised ccr given existing ccr and total of mex + cex, and ccpool
	x <- ex
	if (type == 1) {
		# type 1: just nudge up or down by 1% depending on sign of ex or no change if net demand is zero
		if (ex < 0) {
			# supply exceeds demand so reduce price of CC
			return(ccr / 1.01)
		} else if (ex > 0) {
			# demand exceeds supply so raise price of CC
			return(ccr * 1.01)
		} else {
			return(ccr)
		}
	} else if (type == 2) {
		# type 2: nudge by 1% twice if net demand is more than 2000 either direction
		if (ex < -2000) {
			# supply exceeds demand so reduce price of CC
			return(ccr / 1.01^2)
		} else if (ex < 0) {
			return(ccr / 1.01)
		} else if (ex > 2000) {
			# demand exceeds supply so raise price of CC
			return(ccr * 1.01^2)
		} else if (ex > 0) {
			# demand exceeds supply so raise price of CC
			return(ccr * 1.01)
		} else {
			return(ccr)
		}
	} else if (type == 3) {
		# type 3: nudge is a continuous function of ex, but then modified by two step increase if pool is low
		ccr <- ccr*1.01^(ex/1000)
		if (pool < 2000) {
			return(ccr*1.01^2)
		} else if (pool < 4000) {
			return(ccr*1.01)
		} else {
			return(ccr)
		}
	} else if (type == 4) {
		# type 4: up to a point, nudge is related to ex, with graduated pool adjustment between 0 and 20k
		ex <- min(5000, max(ex, -5000))
		pool <- min(20000, max(0, pool))
		if (ex >= 0 ) {
			return((1.003-0.003*pool/20000)^(ex/1000))
		} else {
			return((1.0+0.003*pool/20000)^(ex/1000))
		}
	} else if (type == 5) {
		# type 5: continuous variation driven by ex only
		ccr <- ccr*1.01^(ex/1000)
		return(ccr)
	} else if (type == 6) {
		# type 6: continuous variation driven by ex only but tighter
		ccr <- ccr*1.01^(ex/500)
		return(ccr)
	} else if (type == 7) {
		# type 7: continuous variation driven by ex only but even tighter
		ccr <- ccr*1.01^(ex/100)
		return(ccr)
	} else if (type == 8) {
		# type 8: continuous variation driven by ex only but much looser
		ccr <- ccr*1.01^(ex/10000)
		return(ccr)
	} else if (type == 9) {
		# type 9: continuous variation driven by ex and pool
		if (pool < 0) {
			ccr <- ccr*1.01^(ex/exc_damp)*(1 + exc_greatest/exc_peak)*(1 - exc_greatest/(exc_width + exc_peak))
		} else if (pool > exc_width) {
			ccr <- ccr*1.01^(ex/exc_damp)*(1 + exc_greatest/(exc_width + exc_peak))*(1 - exc_greatest/exc_peak)
		} else {
			ccr <- ccr*1.01^(ex/exc_damp)*(1 + exc_greatest/(pool + exc_peak))*(1 - exc_greatest/(exc_width - pool + exc_peak))
		}
		return(ccr)
	} else {
		return(ccr)
	}
}


sim_day <- function(d) {  # simulates one day within the current trial
	# start recording data for the day
	days[d,1] <<- d
	days$ccr[d] <<- ccr
	days$ccm[d] <<- ccm
	mn <- 0		# number of merchants participating at the end of the day
	cn <- 0		# number of customers participating at the end of the day
	mcc <- 0		# balances of CC at the end of the day - merchants
	ccc <- 0 		# balances of CC at the end of the day - customers
	ccbal <- 0		# sum of balances of CC > 0
	ccbalcount <- 0	# number of participants with CC balances > 0
	ccballog <- 0	# x log[x] for balances > 0
	dfcpurch <- 0		# FC value of day's purchases made in FC
	dccpurch <- 0		# CC value of day's purchases made in CC
	dccpurchcount <- 0	# number of purchases made in CC in the day
	dccpurchases <- c()	# vector for the CC purchases of the day
	dccpurchatncp <- 0	# day's purchases made in CC, but using FC prices available for those goods
      mex <- 0		# net CC exchanged by merchants at the end of the day +ve is bought
	cex <- 0		# net CC exchanged by customers at the end of the day +ve is bought
	absexc <- 0		# sum of absolute amounts exchanged
	exclog <- 0  	# sum of absolute amounts exchanged x log(of it,2)
	exccount <- 0	# count of exchange amounts
	dccexchanges <- c() # vector of exchange transactions (absolute value)
	mmint <- 0		# CC minted by merchants at the end of the day
	cmint <- 0		# CC minted by customers at the end of the day
	c_sav <- 0		# savings by customer through CC purchases, in FC
	wmean <- 0	# mean (weighted) of factor change ewmas
	wsd <- 0	# standard deviation (weighted) of factor change ewmas

	# merchants consider their policy on CC prices and perhaps change prices
	for (m in 1:m_max) {
		if (merchants$incc[m] == 1) {
			# already using CC
                  # decide whether to stop
			if (runif(1) < 10^(-1*merchants$sent[m]-2)) {
				merchants$incc[m] <<- 0
				merchants$meth[m] <<- "out"
				delete_cc_prices(m)
			} else {
				# revise prices, perhaps
				if (merchants$meth[m] == "now") {
					set_cc_prices(m,ccr*(1-cc_fxl)^0.5)			# Note: using selling price for CC
				} else {
					if (merchants$meth[m] == "weekly" && (d - merchants$prd) %% 7 == 0) {
						set_cc_prices(m,ccr*(1-cc_fxl)^0.5)
					} else {
						# assume monthly
						if (merchants$meth[m] == "monthly" && (d - merchants$prd) %% 30 == 0) {
							set_cc_prices(m,ccr*(1-cc_fxl)^0.5)
						}
					}
				}
			}
		} else {
			# not using CC prices
			if (runif(1) < 10^(merchants$sent[m]-2)) {
				# start setting CC prices
				merchants$incc[m] <<- 1
				merchants$prd[m] <<- d
				if (runif(1) < m_now_prob) {
					merchants$meth[m] <<- "now"
				} else {
					if (runif(1) < m_wkly_prob) {
						merchants$meth[m] <<- "weekly"
					} else {
						merchants$meth[m] <<- "monthly"
					}
				}
				set_cc_prices(m,ccr*(1-cc_fxl)^0.5)				# Note: using selling price for CC
			}
		}
		mn <- mn + merchants$incc[m]
		# take the opportunity to initialise the total of day's sales in CC
		merchants$daysales[m] <<- 0
		merchants$prevccsales[m] <<- merchants$ccsales[m]
		merchants$ccsales[m] <<- 0
	}
	# customers decide whether to be in CC or not
	for (c in 1:c_max) {
		if (customers$incc[c] == 1) {
			# customer is already participating in CC
			if (runif(1) < 10^(-1*customers$sent[c]-2)) {
				customers$incc[c] <<- 0
			}
		} else {
			# customer is not participating in CC
			if (runif(1) < 10^(customers$sent[c]-2)) {
				customers$incc[c] <<- 1
			}
		}
		cn <- cn + customers$incc[c]
		# take the opportunity to initialise daily savings for customers
		customers$prevccsavings[c] <<- customers$ccsavings[c]
		customers$ccsavings[c] <<- 0
	}
      # customers make their purchases
	for (c in 1:c_max) {
		c_sav <- 0
		# number of purchases to make
		pc <- purchase_count(customers$ppd[c], customers$ppd[1] * 3)
		for (p in 1:pc) {
			# make a purchase
			gn <- floor(runif(1, min=1, max=length(goods$id)+1))	# select a random good
			np <- goods$ncp[gn]  	# FC price
			if (customers$incc[c] == 1) {
				# participating in CC so looking for attractive CC prices
				if (goods$ccp[gn] > 0) {
					# has a CC price; is it attractive?
					cp <- goods$ccp[gn]
					if (np/ccr > cp | np/ccm > cp) {	# this uses the mid price, avoiding conflicting conclusions
						# CC price is more attractive; transfer CC to merchant
						customers$cc[c] <<- customers$cc[c] - cp
						# update merchant's stock of CC
						merchants$cc[goods$mid[gn]] <<- merchants$cc[goods$mid[gn]] + cp
						# add to merchant's total of sales for the day
						merchants$daysales[goods$mid[gn]] <<- merchants$daysales[goods$mid[gn]] + cp*ccr
						merchants$ccsales[goods$mid[gn]] <<- merchants$ccsales[goods$mid[gn]] + cp
						# update various stats
						dccpurch <- dccpurch + cp
						dccpurchcount <- dccpurchcount + 1
						dccpurchatncp <- dccpurchatncp + np
						c_sav <- c_sav + (np - cp*ccr)
						customers$ccsavings[c] <<- customers$ccsavings[c] + (np - cp*ccr)
						# add to day's purchases
						dccpurchases[dccpurchcount] <- cp
					} else {
						# buy in FC
						dfcpurch <- dfcpurch + np
					}
				}
			} else {
				# not participating, so simply buy in FC
				dfcpurch <- dfcpurch + np
			}
		}
	}
	# some of the daily stats calculated now
	purchases <<- append(purchases, list(dccpurchases))
	days$merin[d] <<- mn
	days$cusin[d] <<- cn
	days$fcpurch[d] <<- dfcpurch
	days$ccpurch[d] <<- dccpurch
	days$purchRE[d] <<- relative_entropy(purchases[[d]])
	if (dccpurchcount == 0) {
		days$purchbyRE[d] <<- 0
	} else {
		days$purchbyRE[d] <<- days$ccpurch[d]*days$purchRE[d]
	}
	if (dccpurchcount == 0) {
		days$purchGI[d] <<- NaN
		days$purchbyGI[d] <<- 0
	} else {
		days$purchGI[d] <<- 1 - ineq(purchases[[d]],type="Gini")
		days$purchbyGI[d] <<- days$ccpurch[d]*days$purchGI[d]
	}
	days$ccpurchatncp[d] <<- dccpurchatncp
	days$buzz[d] <<- buzz
	# merchants make sales/purchases of CC for the day
	for (m in 1:m_max) {
		# this is where merchants' perceptions are updated
		if (d == 1) {	# f1: exchange activity and f3: balances held
			merchants$ewmas[[m]][1] <<- 0
			merchants$ewmas[[m]][3] <<- 0
		} else if (d == 2) {
			merchants$ewmas[[m]][1] <<- merchants$lambdas[[m]][1]*days$excbyRE[d-1]
			merchants$ewmas[[m]][3] <<- merchants$lambdas[[m]][3]*days$ccheldbyRE[d-1]
		} else {
			merchants$ewmas[[m]][1] <<- merchants$lambdas[[m]][1]*(days$excbyRE[d-1] - days$excbyRE[d-2]) + (1 - merchants$lambdas[[m]][1])*merchants$ewmas[[m]][1]
			merchants$ewmas[[m]][3] <<- merchants$lambdas[[m]][3]*(days$ccheldbyRE[d-1] - days$ccheldbyRE[d-2]) + (1 - merchants$lambdas[[m]][3])*merchants$ewmas[[m]][3]
		}
		if (d == 1) {	# f2: purchases in CC across the whole market and f4: exchange rate and f5: days sales in CC and f6: buzz
			merchants$ewmas[[m]][2] <<- merchants$lambdas[[m]][2]*days$purchbyRE[d]
			merchants$ewmas[[m]][4] <<- merchants$lambdas[[m]][4]*days$ccr[d]
			merchants$ewmas[[m]][5] <<- merchants$lambdas[[m]][5]*merchants$ccsales[m]
			merchants$ewmas[[m]][6] <<- merchants$lambdas[[m]][6]*days$buzz[d]
		} else {
			merchants$ewmas[[m]][2] <<- merchants$lambdas[[m]][2]*(days$purchbyRE[d] - days$purchbyRE[d-1]) + (1 - merchants$lambdas[[m]][2])*merchants$ewmas[[m]][2]
			merchants$ewmas[[m]][4] <<- merchants$lambdas[[m]][4]*(days$ccr[d] - days$ccr[d-1]) + (1 - merchants$lambdas[[m]][4])*merchants$ewmas[[m]][4]
			merchants$ewmas[[m]][5] <<- merchants$lambdas[[m]][5]*(merchants$ccsales[m] - merchants$prevccsales[m]) + (1 - merchants$lambdas[[m]][5])*merchants$ewmas[[m]][5]
			merchants$ewmas[[m]][6] <<- merchants$lambdas[[m]][6]*(days$buzz[d] - days$buzz[d-1]) + (1 - merchants$lambdas[[m]][6])*merchants$ewmas[[m]][6]
		}
		wmean <- sum(merchants$ewmas[[m]]*merchants$weights[[m]]*fmask)/sum(merchants$weights[[m]]*fmask)
		wsd <- sqrt(sum(merchants$weights[[m]]*fmask*(merchants$ewmas[[m]]-wmean)^2)/sum(merchants$weights[[m]]*fmask))
		pccbest <- 1 - pnorm(0, wmean, wsd)

		# this is where sentiment is adjusted, taking into account the day's sales and current buzz level
		merchants$sent[m] <<- (m_sbfrac*max(-2,min(2,merchants$daysales[m]/m_unitsales)) + (1 - m_sbfrac)*max(-2,min(2,buzz/m_unitbuzz)))*m_lambda + (1 - m_lambda)*merchants$sent[m]

		# now adjust holding of CC
		if (merchants$incc[m] == 1) {
			# merchant is participating
			# here is where the target CC holding is set, based on probability that CC will rise relative to FC
			t <- merchants$cashpot[m]*pccbest
			# t <- merchants$hip[m]/2*merchants$ngo[m]*10^(merchants$sent[m]-1)
			# exchange or mint? Depends if you are buying or selling.

			a <- t - merchants$cc[m]			# amount wanted
			if (a > 0) {		# buying
				if (ccr/(1-cc_fxl)^0.5 <= ccm) {	# exchange is cheaper than minting
					mex <- mex + a
					exccount <- exccount + 1
					dccexchanges[exccount] <- a
				} else {					# minting is cheaper than exchange
					mmint <- mmint + a
				}
			} else if (a < 0) {	# selling, only the exchange is available
				mex <- mex + a
				exccount <- exccount + 1
				dccexchanges[exccount] <- abs(a)
			}
			merchants$cc[m] <<- t
		} else {
			# not participating - so sell them all, if any
			if (merchants$cc[m] != 0) {
				mex <- mex - merchants$cc[m]
				exccount <- exccount + 1
				dccexchanges[exccount] <- abs(merchants$cc[m])
				merchants$cc[m] <<- 0
			}
		}
		mcc <- mcc + merchants$cc[m]
		if (merchants$cc[m] > 0) {
			ccbal <- ccbal + merchants$cc[m]
			ccbalcount <- ccbalcount + 1
		}
	}
      # customers make purchases/sales of CC for the day
	for (c in 1:c_max) {
		# this is where customers' perceptions are updated
		if (d == 1) {	# f1: exchange activity and f3: balances held
			customers$ewmas[[c]][1] <<- 0
			customers$ewmas[[c]][3] <<- 0
		} else if (d == 2) {
			customers$ewmas[[c]][1] <<- customers$lambdas[[c]][1]*days$excbyRE[d-1]
			customers$ewmas[[c]][3] <<- customers$lambdas[[c]][3]*days$ccheldbyRE[d-1]
		} else {
			customers$ewmas[[c]][1] <<- customers$lambdas[[c]][1]*(days$excbyRE[d-1] - days$excbyRE[d-2]) + (1 - customers$lambdas[[c]][1])*customers$ewmas[[c]][1]
			customers$ewmas[[c]][3] <<- customers$lambdas[[c]][3]*(days$ccheldbyRE[d-1] - days$ccheldbyRE[d-2]) + (1 - customers$lambdas[[c]][3])*customers$ewmas[[c]][3]
		}
		if (d == 1) {	# f2: purchases in CC across the whole market and f4: exchange rate and f5: days sales in CC and f6: buzz
			customers$ewmas[[c]][2] <<- customers$lambdas[[c]][2]*days$purchbyRE[d]
			customers$ewmas[[c]][4] <<- customers$lambdas[[c]][4]*days$ccr[d]
			customers$ewmas[[c]][5] <<- customers$lambdas[[c]][5]*customers$ccsavings[c]
			customers$ewmas[[c]][6] <<- customers$lambdas[[c]][6]*days$buzz[d]
		} else {
			customers$ewmas[[c]][2] <<- customers$lambdas[[c]][2]*(days$purchbyRE[d] - days$purchbyRE[d-1]) + (1 - customers$lambdas[[c]][2])*customers$ewmas[[c]][2]
			customers$ewmas[[c]][4] <<- customers$lambdas[[c]][4]*(days$ccr[d] - days$ccr[d-1]) + (1 - customers$lambdas[[c]][4])*customers$ewmas[[c]][4]
			customers$ewmas[[c]][5] <<- customers$lambdas[[c]][5]*(customers$ccsavings[c] - customers$prevccsavings[c]) + (1 - customers$lambdas[[c]][5])*customers$ewmas[[c]][5]
			customers$ewmas[[c]][6] <<- customers$lambdas[[c]][6]*(days$buzz[d] - days$buzz[d-1]) + (1 - customers$lambdas[[c]][6])*customers$ewmas[[c]][6]
		}
		wmean <- sum(customers$ewmas[[c]]*customers$weights[[c]])/sum(customers$weights[[c]])
		wsd <- sqrt(sum(customers$weights[[c]]*(customers$ewmas[[c]]-wmean)^2)/sum(customers$weights[[c]]))
		pccbest <- 1 - pnorm(0, wmean, wsd)

		# this is where sentiment is adjusted, taking into account the day's savings and current buzz level
		customers$sent[c] <<- (c_sbfrac*max(-2,min(2,c_sav/c_unitsav)) + (1 - c_sbfrac)*max(-2,min(2,buzz/c_unitbuzz)))*c_lambda + (1 - c_lambda)*customers$sent[c]
		# now adjust holding of CC
		if (customers$incc[c] == 1) {
			# customer is participating in CC
			# here is where the target CC holding is set, based on probability that CC will rise relative to FC
			t <- customers$cashpot[c]*pccbest
			# t <- customers$ppd[c]*aveprice*10^customers$sent[c]	# this is where sentiment affects target stock of CC
			# exchange or mint? It depends on whether you are buying or selling.
			a <- t - customers$cc[c]
			if (a > 0) {		# buying
				if (ccr/(1-cc_fxl)^0.5 <= ccm) {	# exchange is cheaper than minting
					cex <- cex + a
					exccount <- exccount + 1
					dccexchanges[exccount] <- a
				} else {					# minting is cheaper than exchange
					cmint <- cmint + a
				}
			} else if (a < 0) {	# selling, only the exchange is available
				cex <- cex + a
				exccount <- exccount + 1
				dccexchanges[exccount] <- abs(a)
			}
			customers$cc[c] <<- t
		} else {
			# not participating - so sell them all, if any
			if (customers$cc[c] != 0) {
				cex <- cex - customers$cc[c]
				exccount <- exccount + 1
				dccexchanges[exccount] <- abs(customers$cc[c])
				customers$cc[c] <<- 0
			}
		}
		ccc <- ccc + customers$cc[c]
		if (customers$cc[c] > 0) {
			ccbal <- ccbal + customers$cc[c]
			ccbalcount <- ccbalcount + 1
		}
	}
	# adjust pool and cc rate
	ccpool <<- ccpool - mex - cex
	ccr <<- update_ccr(ccr, mex+cex, ccpool, exc_type)

	# add to exchanges list
	exchanges <<- append(exchanges, list(dccexchanges))

	# update day table
	days$pool[d] <<- ccpool
	days$mercc[d] <<- mcc
	days$cuscc[d] <<- ccc
	days$totalcc[d] <<- ccpool + mcc + ccc
	days$heldRE[d] <<- relative_entropy(c(merchants$cc[merchants$cc>0], customers$cc[customers$cc>0]))
	if (ccbalcount == 0) {
		days$ccheldbyRE[d] <<- 0
	} else {
		days$ccheldbyRE[d] <<- (days$mercc[d] + days$cuscc[d])*days$heldRE[d]
	}
	if (ccbalcount == 0) {
		days$heldGI[d] <<- NaN
		days$ccheldbyGI[d] <<- 0
	} else {
		days$heldGI[d] <<- 1 - ineq(c(merchants$cc[merchants$cc>0], customers$cc[customers$cc>0]), type="Gini")
		days$ccheldbyGI[d] <<- (days$mercc[d] + days$cuscc[d])*days$heldGI[d]
	}
	days$velcc[d] <<- days$ccpurch[d]/(days$mercc[d]+days$cuscc[d])
	days$REvelcc[d] <<- days$purchbyRE[d]/days$ccheldbyRE[d]
	days$GIvelcc[d] <<- days$purchbyGI[d]/days$ccheldbyGI[d]
	if (dccpurchatncp > 0) {
		days$fisherccr[d] <<- (days$mercc[d] + days$cuscc[d]) * days$velcc[d] / dccpurchatncp
	} else {
		days$fisherccr[d] <<- "NA"
	}
	days$mermint[d] <<- mmint
	days$cusmint[d] <<- cmint
	days$merccex[d] <<- mex
	days$cusccex[d] <<- cex
	days$excRE[d] <<- relative_entropy(exchanges[[d]])
	if (exccount == 0) {
		days$excbyRE[d] <<- 0
	} else {
		days$excbyRE[d] <<- sum(exchanges[[d]])*days$excRE[d]
	}
	if (exccount == 0) {
		days$excGI[d] <<- NaN
		days$excbyGI[d] <<- 0
	} else {
		days$excGI[d] <<- 1 - ineq(exchanges[[d]],type="Gini")
		days$excbyGI[d] <<- sum(exchanges[[d]])*days$excGI[d]
	}
}


# functions to run a whole trial, simulation, or experiment

run_trial <- function() {	# run a trial, going through the required number of days
	trial_reset()
	for (d in 1:t_days) {
		sim_day(d)
		if (d %% 100 == 0) {
			cat(d,"days done\n")
			flush.console()
		}
	}
}

run_sim <- function() {       # run many trials and collect stats about them
	sim_reset()
	for (t in 1:s_trials) {
		run_trial()
		trials[t,1] <<- t
		trials$poolmax[t] <<- max(days$pool)
		trials$poolmin[t] <<- min(days$pool)
		trials$totalcc[t] <<- days$totalcc[t_days]
		trials$maxccr[t] <<- max(days$ccr)
		trials$minccr[t] <<- min(days$ccr)
		trials$aveccr[t] <<- mean(days$ccr)
		trials$fccr[t] <<- days$ccr[t_days]
		trials$sd[t] <<- sd(days$ccr)
		trials$fd[t] <<- higuchi2(days$ccr, 30)
		trials$runave[t] <<- mean(run_dist(days$ccr))
		trials$diffave[t] <<- mean(abs(days$ccr[2:730]-days$ccr[1:729]))
		trials$poolout[t] <<- ifelse(min(days$pool) < 0, 1, 0)
		if (t %% 10 == 0) {
			cat(t,"trials done\n")
			flush.console()
		}
	}
}

run_expt1 <- function() {	# run many simulations with a factor changed each time
	expt_reset()
	ccr_start <<- 1
	run_sim()
	expts[1,1] <<- 1
	expts[1,2] <<- mean(trials$fccr)
	ccr_start <<- 1.5
	run_sim()
	expts[2,1] <<- 1
	expts[2,2] <<- mean(trials$fccr)
	ccr_start <<- 2
	run_sim()
	expts[3,1] <<- 1
	expts[3,2] <<- mean(trials$fccr)
}


run_expt2 <- function() {	# run many simulations with a factor changed each time
	expt_reset()
	ccm_start <<- 10		# no capping
	t_days <<- 365		# longer term
	m_pot_mult <<- 0.25	# lot cash pots
	c_pot_mult <<- 0.25	# low cash pots
	# condition 1: low proportion of monthly prices
	m_now_prob <<- 0.5
	m_wkly_prob <<- 0.5
	run_sim()
	expts[1,1] <<- 1
	expts[1,2] <<- mean(trials$fccr)
	trials1 <<- trials
	# condition 2: high proportion of monthly prices
	m_now_prob <<- 0.05
	m_wkly_prob <<- 0.05
	run_sim()
	expts[2,1] <<- 1
	expts[2,2] <<- mean(trials$fccr)
	trials2 <<- trials
}


